<?php

namespace App\Database\Seeds;

use CodeIgniter\Database\Seeder;
use CodeIgniter\CLI\CLI;

class DbipCityIpv4Seeder extends Seeder
{
    protected string $csvPath   = 'dbip-city-ipv4.csv';
    protected int $batchSize    = 1000;
    protected string $table     = 'dbip_city_ipv4';

    /** CSV has no header; 10 columns with empty at indices 4, 6, 9. We use: 0,1,2,3,5,7,8. */
    protected array $csvIndices = [0 => 'start_ip', 1 => 'end_ip', 2 => 'country_code', 3 => 'region', 5 => 'city', 7 => 'latitude', 8 => 'longitude'];

    public function run(): void
    {
        $path = ROOTPATH . $this->csvPath;

        if (! is_file($path)) {
            if (is_cli()) {
                CLI::error("CSV file not found: {$path}");
            }
            throw new \RuntimeException("CSV file not found: {$path}");
        }

        $handle = fopen($path, 'r');
        if ($handle === false) {
            throw new \RuntimeException("Could not open CSV: {$path}");
        }

        $tableColumns = $this->db->getFieldNames($this->table);
        if ($tableColumns === []) {
            fclose($handle);
            throw new \RuntimeException("Table {$this->table} not found. Run migration first.");
        }

        $builder = $this->db->table($this->table);
        $batch   = [];
        $total   = 0;

        if (is_cli()) {
            CLI::write("Importing {$this->csvPath} into {$this->table}...", 'cyan');
        }

        while (($row = fgetcsv($handle)) !== false) {
            if (count($row) < 9) {
                continue;
            }
            $data = [];
            foreach ($this->csvIndices as $index => $col) {
                $data[$col] = $row[$index] ?? '';
            }
            $data = array_intersect_key($data, array_flip($tableColumns));
            $batch[] = $data;

            if (count($batch) >= $this->batchSize) {
                $builder->insertBatch($batch);
                $total += count($batch);
                $batch = [];
                if (is_cli()) {
                    CLI::print('.');
                }
            }
        }

        fclose($handle);

        if ($batch !== []) {
            $builder->insertBatch($batch);
            $total += count($batch);
        }

        if (is_cli()) {
            CLI::newLine();
            CLI::write("Imported {$total} rows into {$this->table}.", 'green');
        }
    }
}
