<?php

namespace App\Database\Seeds;

use CodeIgniter\Database\Seeder;
use CodeIgniter\CLI\CLI;

class GeolocationSeeder extends Seeder
{
    /**
     * Path to the CSV file (project root).
     */
    protected string $csvPath = 'geolocationDatabaseIPv4.csv';

    /**
     * Number of rows to insert per batch.
     */
    protected int $batchSize = 1000;

    /**
     * CSV column order (must match CSV header).
     */
    protected array $columns = [
        'ip_version',
        'start_ip',
        'end_ip',
        'continent',
        'country_code',
        'country',
        'state',
        'city',
        'zip',
        'timezone',
        'latitude',
        'longitude',
        'accuracy',
        'source',
    ];

    public function run(): void
    {
        $path = ROOTPATH . $this->csvPath;

        if (! is_file($path)) {
            if (is_cli()) {
                CLI::error("CSV file not found: {$path}");
            }
            throw new \RuntimeException("CSV file not found: {$path}");
        }

        $handle = fopen($path, 'r');
        if ($handle === false) {
            throw new \RuntimeException("Could not open CSV: {$path}");
        }

        // Skip header row
        $header = fgetcsv($handle);
        if ($header === false) {
            fclose($handle);
            throw new \RuntimeException('CSV file is empty or invalid.');
        }

        $table = 'geolocationDatabaseIPv4';
        $tableColumns = $this->db->getFieldNames($table);
        if ($tableColumns === []) {
            fclose($handle);
            throw new \RuntimeException("Table {$table} not found or has no columns.");
        }
        $builder = $this->db->table($table);
        $batch = [];
        $total = 0;

        if (is_cli()) {
            CLI::write("Importing {$this->csvPath} into {$table}...", 'cyan');
        }

        while (($row = fgetcsv($handle)) !== false) {
            if (count($row) < count($this->columns)) {
                continue;
            }
            $data = array_combine($this->columns, array_slice($row, 0, count($this->columns)));
            if ($data === false) {
                continue;
            }
            // Only insert columns that exist in the table (table columns now match CSV names after migration)
            $batch[] = array_intersect_key($data, array_flip($tableColumns));

            if (count($batch) >= $this->batchSize) {
                $builder->insertBatch($batch);
                $total += count($batch);
                $batch = [];
                if (is_cli()) {
                    CLI::print('.');
                }
            }
        }

        fclose($handle);

        if ($batch !== []) {
            $builder->insertBatch($batch);
            $total += count($batch);
        }

        if (is_cli()) {
            CLI::newLine();
            CLI::write("Imported {$total} rows into {$table}.", 'green');
        }
    }
}
