<?php

namespace App\Database\Seeds;

use CodeIgniter\Database\Seeder;
use CodeIgniter\CLI\CLI;

class Ip2locationLiteDb11Seeder extends Seeder
{
    protected string $csvPath   = 'IP2LOCATION-LITE-DB11.CSV';
    protected int $batchSize    = 1000;
    protected string $table     = 'ip2location_lite_db11';

    /** CSV has no header; column order. */
    protected array $columns = [
        'ip_from',
        'ip_to',
        'country_code',
        'country_name',
        'region',
        'city',
        'latitude',
        'longitude',
        'zip_code',
        'time_zone',
    ];

    public function run(): void
    {
        $path = ROOTPATH . $this->csvPath;

        if (! is_file($path)) {
            if (is_cli()) {
                CLI::error("CSV file not found: {$path}");
            }
            throw new \RuntimeException("CSV file not found: {$path}");
        }

        $handle = fopen($path, 'r');
        if ($handle === false) {
            throw new \RuntimeException("Could not open CSV: {$path}");
        }

        $tableColumns = $this->db->getFieldNames($this->table);
        if ($tableColumns === []) {
            fclose($handle);
            throw new \RuntimeException("Table {$this->table} not found. Run migration first.");
        }

        $builder = $this->db->table($this->table);
        $batch   = [];
        $total   = 0;

        if (is_cli()) {
            CLI::write("Importing {$this->csvPath} into {$this->table}...", 'cyan');
        }

        while (($row = fgetcsv($handle)) !== false) {
            if (count($row) < count($this->columns)) {
                continue;
            }
            $data = array_combine($this->columns, array_slice($row, 0, count($this->columns)));
            if ($data === false) {
                continue;
            }
            // IP2Location uses "-" for empty; normalize to empty string
            foreach ($data as $k => $v) {
                if (trim((string) $v) === '-') {
                    $data[$k] = '';
                }
            }
            $data = array_intersect_key($data, array_flip($tableColumns));
            $batch[] = $data;

            if (count($batch) >= $this->batchSize) {
                $builder->insertBatch($batch);
                $total += count($batch);
                $batch = [];
                if (is_cli()) {
                    CLI::print('.');
                }
            }
        }

        fclose($handle);

        if ($batch !== []) {
            $builder->insertBatch($batch);
            $total += count($batch);
        }

        if (is_cli()) {
            CLI::newLine();
            CLI::write("Imported {$total} rows into {$this->table}.", 'green');
        }
    }
}
